<?php

namespace App\Filament\Widgets;

use App\Models\Event;
use Filament\Forms;
use Illuminate\Database\Eloquent\Model;
use Saade\FilamentFullCalendar\Widgets\FullCalendarWidget;
use Saade\FilamentFullCalendar\Actions;
use Saade\FilamentFullCalendar\Data\EventData;

class CalendarWidget extends FullCalendarWidget
{
    public Model | string | null $model = Event::class;

    public function fetchEvents(array $fetchInfo): array
    {
        return Event::query()
            ->where('start', '>=', $fetchInfo['start'])
            ->where('end', '<=', $fetchInfo['end'])
            ->get()
            ->map(
                fn (Event $event) => EventData::make()
                    ->id($event->id)
                    ->title($event->title)
                    ->start($event->start)
                    ->end($event->end)
            )
            ->toArray();
    }

    public function getFormSchema(): array
    {
        return [
            Forms\Components\TextInput::make('title')
                ->label('Titolo')
                ->required(),
            Forms\Components\Textarea::make('description')
                ->label('Descrizione'),
            Forms\Components\DateTimePicker::make('start')
                ->label('Inizio')
                ->required(),
            Forms\Components\DateTimePicker::make('end')
                ->label('Fine')
                ->required(),
        ];
    }

    protected function headerActions(): array
    {
        return [
            Actions\CreateAction::make()
                ->mountUsing(function (Forms\Form $form, array $arguments) {
                    $form->fill([
                        'start' => $arguments['start'] ?? null,
                        'end' => $arguments['end'] ?? null
                    ]);
                }),
        ];
    }

    protected function modalActions(): array
    {
        return [
            Actions\EditAction::make()
                ->mountUsing(function (Event $record, Forms\Form $form, array $arguments) {
                    $form->fill([
                        'title' => $record->title,
                        'description' => $record->description,
                        'start' => $arguments['event']['start'] ?? $record->start,
                        'end' => $arguments['event']['end'] ?? $record->end
                    ]);
                }),
            Actions\DeleteAction::make(),
        ];
    }

    public function config(): array
    {
        return [
            'editable' => true,
            'selectable' => true,
            'firstDay' => 1,
            'headerToolbar' => [
                'left' => 'dayGridMonth,dayGridWeek,dayGridDay, list',
                'center' => 'title',
                'right' => 'prev,next today',
            ],

        ];
    }

    public function eventDidMount(): string
    {
        return <<<JS
            function({ event, el }) {
                el.setAttribute("x-tooltip", "tooltip");
                el.setAttribute("x-data", "{ tooltip: '" + event.title + "' }");
            }
        JS;
    }
}
