<?php

namespace App\Filament\Resources;

use App\Filament\Resources\ContrattoTemplateResource\Pages;
use App\Models\ContrattoTemplate;
use Filament\Forms\Components\RichEditor;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Actions\Action;
use App\Filament\Resources\ContrattoTemplateResource\RelationManagers\ContrattiRelationManager;

class ContrattoTemplateResource extends Resource
{
    protected static ?string $model = ContrattoTemplate::class;
    protected static ?string $navigationIcon = 'heroicon-o-document-text';
    protected static ?string $navigationGroup = 'Contratti';
    protected static ?string $navigationLabel = 'Template contratti';
    protected static ?int $navigationSort = 3;

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                TextInput::make('nome')
                    ->required()
                    ->disabled(),
                TextInput::make('descrizione')
                    ->disabled(),
                RichEditor::make('contenuto')
                    ->disabled()
                    ->columnSpanFull(),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                TextColumn::make('nome')
                    ->searchable()
                    ->sortable(),
                TextColumn::make('descrizione')
                    ->searchable()
                    ->wrap(),
                TextColumn::make('created_at')
                    ->label('Data creazione')
                    ->dateTime('d/m/Y H:i')
                    ->sortable(),
                TextColumn::make('contratti_count')
                    ->label('Contratti che usano questo template')
                    ->counts('contratti'),
            ])
            ->filters([
                Tables\Filters\Filter::make('custom')
                    ->label('Solo template personalizzati')
                    ->query(function (Builder $query) {
                        return $query->where(function($q) {
                            $q->where(function($inner) {
                                // Template creati automaticamente
                                $inner->where('nome', 'like', 'Template Personalizzato%')
                                    ->orWhere('descrizione', 'like', 'Template personalizzato%')
                                    ->orWhere('descrizione', 'like', '%creato manualmente%');
                            })
                            ->orWhere(function($inner) {
                                // Escludi i template predefiniti
                                $inner->whereNotIn('nome', [
                                    'Template Personale Strutturato',
                                    'Template Docenza',
                                    'Template Tutoraggio',
                                    'Template Lavoro Dipendente',
                                    'Template Collaborazione Occasionale',
                                    'Template COCOCO',
                                    'Template Collaborazione Autonoma'
                                ]);
                            });
                        });
                    })
                    ->default(),
            ])
            ->actions([
                Action::make('view')
                    ->label('Visualizza')
                    ->icon('heroicon-m-eye')
                    ->url(fn (ContrattoTemplate $record) => static::getUrl('view', ['record' => $record])),
                Tables\Actions\DeleteAction::make()
                    ->visible(fn (ContrattoTemplate $record) => 
                        // Permetti eliminazione solo se:
                        // 1. È un template personalizzato
                        // 2. Non è usato da nessun contratto
                        str_contains($record->descrizione, 'Template personalizzato') && 
                        $record->contratti()->count() === 0
                    )
                    ->before(function (ContrattoTemplate $record) {
                        if ($record->contratti()->count() > 0) {
                            return false;
                        }
                    }),
            ])
            ->bulkActions([])
            ->defaultSort('created_at', 'desc');
    }

    public static function getRelations(): array
    {
        return [
            ContrattiRelationManager::class,
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListContrattoTemplates::route('/'),
            'view' => Pages\ViewContrattoTemplate::route('/{record}'),
        ];
    }

    

    public static function getEloquentQuery(): Builder
    {
        return parent::getEloquentQuery();
    }
} 