<?php

namespace App\Filament\Pages;

use App\Models\Anagrafica as AnagraficaModel;
use Filament\Pages\Page;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Concerns\InteractsWithTable;
use Filament\Tables\Contracts\HasTable;
use Filament\Tables\Table;
use Filament\Tables\Actions\Action;
use Filament\Tables\Actions\DeleteAction;
use Filament\Tables\Columns\IconColumn;
use Filament\Tables\Actions\DeleteBulkAction;
use Filament\Tables\Filters\SelectFilter;
use Filament\Tables\Grouping\Group;
use Filament\Tables\Columns\ToggleColumn;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Auth;
use App\Models\User;
use Illuminate\Support\Str;
use Filament\Notifications\Notification;
use App\Notifications\PanelAccessCredentials;
use BezhanSalleh\FilamentShield\Traits\HasPageShield;
use Filament\Tables\Actions\TableAction;
use Filament\Tables;

class GestioneContatti extends Page implements HasTable
{
    use InteractsWithTable;
    use HasPageShield;

    protected static ?string $navigationGroup = 'Anagrafica';
    protected static ?string $navigationIcon = 'heroicon-o-users';
    protected static ?string $navigationLabel = 'Anagrafiche';
    protected static ?string $title = 'Gestione Contatti';
    protected static string $view = 'filament.pages.gestione-contatti';
    protected static ?int $navigationSort = 2;

    public function table(Table $table): Table
    {
        return $table
            ->query(AnagraficaModel::query())
            ->groups([
                Group::make('tipologia_utente')
                    ->label('Tipologia Utente')
                    ->getTitleFromRecordUsing(fn ($record) => $record->tipologia_utente ?? 'Non specificata')
                    ->collapsible(),
            ])
            ->defaultGroup('tipologia_utente')
            ->groupingSettingsInDropdownOnDesktop()
            ->columns([
                TextColumn::make('nome')->sortable()->searchable(),
                TextColumn::make('cognome')->sortable()->searchable(),
                TextColumn::make('email')->sortable()->searchable(),
                ToggleColumn::make('has_panel_access')
                    ->label('Accesso Pannello')
                    ->onColor('success')
                    ->offColor('danger')
                    ->afterStateUpdated(function ($record, $state) {
                        try {
                            if ($state) {
                                // Se l'accesso viene attivato
                                if (!$record->panel_email) {
                                    $record->panel_email = $record->email;
                                }
                                
                                // Genera una password casuale
                                $password = Str::random(12);
                                
                                // Crea o aggiorna l'utente
                                $user = User::updateOrCreate(
                                    ['email' => $record->panel_email],
                                    [
                                        'name' => $record->nome_completo,
                                        'password' => Hash::make($password),
                                    ]
                                );

                                // Assegna il ruolo utente
                                $user->syncRoles(['utente']);
                                
                                $record->panel_access_created_at = now();
                                $record->save();

                                // Invia email con le credenziali
                                $user->notify(new PanelAccessCredentials($record->nome_completo, $password));
                                
                                // Notifica nel pannello admin
                                Notification::make()
                                    ->title('Accesso al pannello attivato')
                                    ->success()
                                    ->body("Email inviata a: {$record->panel_email}\nPassword temporanea: {$password}")
                                    ->persistent()
                                    ->send();
                            } else {
                                // Se l'accesso viene disattivato
                                if ($record->panel_email) {
                                    User::where('email', $record->panel_email)->delete();
                                    $record->panel_access_created_at = null;
                                    $record->save();
                                    
                                    Notification::make()
                                        ->title('Accesso al pannello disattivato')
                                        ->warning()
                                        ->send();
                                }
                            }
                        } catch (\Exception $e) {
                            Notification::make()
                                ->title('Errore')
                                ->danger()
                                ->body('Si è verificato un errore durante l\'operazione.')
                                ->persistent()
                                ->send();

                            // Ripristina lo stato del toggle
                            $record->has_panel_access = !$state;
                            $record->save();
                        }
                    }),
                IconColumn::make('is_complete')
                    ->label('Anagrafica Completa')
                    ->alignCenter()
                    ->boolean()
                    ->trueIcon('heroicon-o-check-circle')
                    ->falseIcon('heroicon-o-x-circle')
                    ->trueColor('success')
                    ->falseColor('danger')
                    ->getStateUsing(fn ($record) => $record->isComplete()),
                IconColumn::make('status')
                    ->label('Stato Invito')
                    ->alignCenter()
                    ->getStateUsing(function ($record) {
                        if ($record->is_complete) {
                            if ($record->invited_at) {
                                return 'completed_by_invite';
                            }
                            return 'completed_manually';
                        } elseif ($record->invited_at) {
                            return 'invited';
                        } else {
                            return 'not_invited';
                        }
                    })
                    ->options([
                        'heroicon-o-check-circle' => 'completed_by_invite',
                        'heroicon-o-user-circle' => 'completed_manually',
                        'heroicon-o-envelope' => 'invited',
                        'heroicon-o-x-circle' => 'not_invited',
                    ])
                    ->colors([
                        'success' => 'completed_by_invite',
                        'info' => 'completed_manually',
                        'warning' => 'invited',
                        'danger' => 'not_invited',
                    ])
                    ->tooltip(function ($record) {
                        if ($record->is_complete) {
                            if ($record->invited_at) {
                                return 'Anagrafica completata tramite invito';
                            }
                            return 'Anagrafica inserita manualmente';
                        } elseif ($record->invited_at) {
                            return 'Invito inviato';
                        } else {
                            return 'Non invitato';
                        }
                    }),
            ])
            ->filters([
                SelectFilter::make('tipologia_utente')
                    ->label('Tipologia Utente')
                    ->options([
                        'Personale Docente' => 'Personale Docente',
                        'Personale Tecnico Amministrativo' => 'Personale Tecnico Amministrativo',
                        'Collaboratori' => 'Collaboratori',
                    ])
                    ->multiple()
                    ->searchable()
            ])
            ->actions([
                Tables\Actions\Action::make('nulla_osta')
                    ->label('Nulla Osta')
                    ->icon('heroicon-o-document-text')
                    ->color('success')
                    ->url(fn ($record) => route('nulla-osta.download', $record))
                    ->openUrlInNewTab()
                    ->tooltip('Scarica il nulla osta per il Ministero'),
                Tables\Actions\Action::make('edit')
                    ->label('Modifica')
                    ->url(function (AnagraficaModel $record) {
                        return Anagrafica::getUrl(['id' => $record->id]);
                    })
                    ->icon('heroicon-o-pencil'),
                Tables\Actions\DeleteAction::make('delete')
                    ->label('Elimina')
                    ->icon('heroicon-o-trash')
                    ->requiresConfirmation()
            ])
            ->bulkActions([
                Tables\Actions\DeleteBulkAction::make(),
            ])
            ->headerActions([
                Tables\Actions\Action::make('create')
                    ->label('Nuova Anagrafica')
                    ->url(fn () => Anagrafica::getUrl())
                    ->icon('heroicon-o-plus')
            ]);
    }
}