<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Facades\Storage;

use App\Models\TipoContratto;

class Contratto extends Model
{
    use HasFactory;
    use SoftDeletes;

    /**
     * The table associated with the model.
     *
     * @var string
     */
    protected $table = 'contratti';

    /**
     * The attributes that are mass assignable.
     *
     * @var array
     */
    protected $fillable = [
        'project_id',
        'anagrafica_id',
        'tipo_contratto_id',
        'data_inizio',
        'data_fine',
        'lavoro_subordinato_pa',
        'denominazione_pa',
        'indirizzo_pa',
        'esclusione_soggettiva',
        'tipo_esclusione',
        'altra_esclusione',
        'numero_autorizzazione',
        'data_autorizzazione',
        'ente_autorizzazione',
        'titolare_pensione_diretta',
        'non_iscritto_previdenza',
        'pdf_path',
        'ore_totali',
        'compenso',
        'costo_lordo_aziendale',
        'costo_lordo_dipendente',
        'costo_orario',
        'costo_giornaliero',
        'template_id',
        'contenuto_custom',
        'descrizione_incarico',
        'ore_progetto',
        'yousign_procedure_id',
        'yousign_signer_id',
        'yousign_status',
        'signed_at',
        'is_uploaded',
        'secondo_firmatario',
        'numero_decreto',
        'numero_posizioni',
        'regime_tempo',
        'gsd',
        'ssd',
        'numero_decreto_dg',
        'data_delibera_dipartimento',
        'data_delibera_cda',
        'corso_id',
        'cfu',
        'semestre',
        'anno',
        'classe',
        'anno_accademico',
        'profilo',
        'mansioni',
        'tipo_tempo',
        'tipo_orario',
        'livello',
        'ore_settimanali',
        'retribuzione_mensile',
        'retribuzione_annua',
        'macrosettore',
        'settore_concorsuale',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array
     */
    protected $casts = [
        'data_inizio' => 'date',
        'data_fine' => 'date',
        'lavoro_subordinato_pa' => 'boolean',
        'esclusione_soggettiva' => 'boolean',
        'data_autorizzazione' => 'date',
        'titolare_pensione_diretta' => 'boolean',
        'non_iscritto_previdenza' => 'boolean',
        'compenso' => 'decimal:2',
        'costo_lordo_aziendale' => 'decimal:2',
        'costo_lordo_dipendente' => 'decimal:2',
        'costo_orario' => 'decimal:2',
        'costo_giornaliero' => 'decimal:2',
        'project_data_inizio' => 'date',
        'project_data_fine' => 'date',
        'signed_at' => 'datetime',
        'data_delibera_dipartimento' => 'date',
        'data_delibera_cda' => 'date',
        'is_uploaded' => 'boolean',
        'retribuzione_mensile' => 'decimal:2',
        'retribuzione_annua' => 'decimal:2',
    ];

    public function anagrafica()
    {
        return $this->belongsTo(Anagrafica::class)->withTrashed();
    }

    public function tipoContratto()
    {   
        return $this->belongsTo(TipoContratto::class);
    }

    public function corsi()
    {
        return $this->belongsToMany(Corso::class, 'contratto_corso');
    }

    public function project()
    {
        return $this->belongsTo(Project::class);
    }

    public function template()
    {
        return $this->belongsTo(ContrattoTemplate::class, 'template_id');
    }

    public function secondoFirmatario()
    {
        return $this->belongsTo(Firmatario::class, 'secondo_firmatario');
    }

    public function corso()
    {
        return $this->belongsTo(Corso::class, 'corso_id');
    }

    protected static function boot()
    {
        parent::boot();

        // Quando un contratto viene eliminato
        static::deleted(function ($contratto) {
            // Elimina l'evento correlato
            Event::where('title', 'Scadenza contratto: ' . $contratto->anagrafica->nome_completo)
                ->delete();
        });

        // Quando un contratto viene aggiornato
        static::updating(function ($contratto) {
            // Se il tipo di contratto è cambiato
            if ($contratto->isDirty('tipo_contratto_id')) {
                // Elimina il PDF esistente
                if ($contratto->pdf_path && Storage::disk('public')->exists($contratto->pdf_path)) {
                    Storage::disk('public')->delete($contratto->pdf_path);
                }
                // Resetta il campo pdf_path
                $contratto->pdf_path = null;
                // Resetta lo stato di firma
                $contratto->yousign_procedure_id = null;
                $contratto->yousign_signer_id = null;
                $contratto->yousign_status = null;
                $contratto->signed_at = null;
            }
        });
    }

}