<?php

namespace App\Livewire;

use Livewire\Component;

use App\Models\Comune;
use App\Models\Anagrafica as AnagraficaModel;
use App\Models\Provincia;

use App\Forms\Components\GooglePlacesAutocomplete;
use App\Mail\InviteToCompleteAnagrafica;

use robertogallea\LaravelCodiceFiscale\CodiceFiscale;

use Filament\Notifications\Notification;
use Filament\Actions\Concerns\InteractsWithActions;
use Filament\Actions\Contracts\HasActions;

use Filament\Forms\Form;
use Filament\Forms\Get;
use Filament\Forms\Components\DatePicker;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Tabs;
use Filament\Forms\Components\Toggle;
use Filament\Forms\Concerns\InteractsWithForms;
use Filament\Forms\Contracts\HasForms;

use Illuminate\Support\Str;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Storage;

use Danielebarbaro\LaravelVatEuValidator\Rules\VatNumber;
use Danielebarbaro\LaravelVatEuValidator\Facades\VatValidatorFacade;

use GuzzleHttp\Client;

class CreateAnagrafica extends Component implements HasForms, HasActions
{
    use InteractsWithForms;
    use InteractsWithActions;
    
    public ?array $data = [];
    public AnagraficaModel $anagrafica;
    public $isEditing = false;
    public $invite_user = false;

    public function mount($id = null)
    {
        $this->invite_user = false;
        
        if ($id) {
            $this->anagrafica = AnagraficaModel::with('domicilio')->findOrFail($id);
            $data = $this->anagrafica->toArray();
            
            // Imposta l'indirizzo di residenza completo
            $data['indirizzo_residenza'] = $this->anagrafica->indirizzo_residenza;
            
            // Se esiste un domicilio, carica i dati nei campi del form
            if ($this->anagrafica->domicilio) {
                $data['has_domicilio'] = true;
                // Imposta l'indirizzo completo per il campo di Google Places
                $data['indirizzo_domicilio'] = $this->anagrafica->domicilio->indirizzo;
                $data['numero_civico_domicilio'] = $this->anagrafica->domicilio->numero_civico;
                $data['cap_domicilio'] = $this->anagrafica->domicilio->cap;
                $data['citta_domicilio'] = $this->anagrafica->domicilio->citta;
                $data['provincia_domicilio'] = $this->anagrafica->domicilio->provincia;
            }

            $this->form->fill($data);
            $this->isEditing = true;
        } else {
            $this->anagrafica = new AnagraficaModel();
            $this->form->fill();
        }
    }
    
    public function findProvinciaId($abbreviazione)
    {
        $provincia = Provincia::where('abbreviazione', $abbreviazione)->first();
        return $provincia ? $provincia->abbreviazione : null;
    }

    private function sendInvitation($data)
    {   
        $token = Str::random(64);
        
        try {
            // Crea un'anagrafica parziale
            $anagrafica = AnagraficaModel::create([
                'nome' => $data['nome'],
                'cognome' => $data['cognome'],
                'email' => $data['email'],
                'telefono' => $data['telefono'],
                'tipologia_utente' => $data['tipologia_utente'],
                'invitation_token' => $token,
                'invited_at' => now(),
                'is_complete' => false,  
            ]);
            
            // Invia l'email di invito
            $url = route('complete.anagrafica', $token);
            Mail::to($data['email'])->send(new InviteToCompleteAnagrafica($anagrafica, $url));
            
            return 'Invito inviato con successo';
        } catch (\Exception $e) {
            throw $e;
        }
    }

    protected function performSave($data): void
    {
        // Prima controlla se esiste un'anagrafica soft-deleted
        $existingSoftDeleted = AnagraficaModel::withTrashed()
            ->where('email', $data['email'])
            ->whereNotNull('deleted_at')
            ->first();

        // Controlla se esiste un'anagrafica parziale non completata
        $existingPartial = AnagraficaModel::where('email', $data['email'])
            ->where('is_complete', false)
            ->first();

        if ($existingSoftDeleted) {
            $existingSoftDeleted->restore();
            
            if (isset($data['invite_user']) && $data['invite_user'] === true) {
                // Se è un invito, aggiorna solo i campi necessari e invia l'invito
                $token = Str::random(64);
                $existingSoftDeleted->update([
                    'nome' => $data['nome'],
                    'cognome' => $data['cognome'],
                    'email' => $data['email'],
                    'telefono' => $data['telefono'],
                    'tipologia_utente' => $data['tipologia_utente'],
                    'invitation_token' => $token,
                    'invited_at' => now(),
                    'is_complete' => false,
                ]);

                try {
                    $url = route('complete.anagrafica', $token);
                    Mail::to($data['email'])->send(new InviteToCompleteAnagrafica($existingSoftDeleted, $url));
                    $message = 'Invito inviato con successo';
                } catch (\Exception $e) {
                    $message = 'Errore durante l\'invio dell\'invito: ' . $e->getMessage();
                }
            } else {
                // Se non è un invito, aggiorna normalmente
                $existingSoftDeleted->update($data);
                
                // Gestione del domicilio
                if (!empty($data['has_domicilio'])) {
                    $domicilioData = [
                        'indirizzo' => $data['indirizzo_domicilio'],
                        'numero_civico' => $data['numero_civico_domicilio'],
                        'cap' => $data['cap_domicilio'],
                        'citta' => $data['citta_domicilio'],
                        'provincia' => $data['provincia_domicilio'],
                    ];
                    
                    $existingSoftDeleted->domicilio()->updateOrCreate(
                        ['anagrafica_id' => $existingSoftDeleted->id],
                        $domicilioData
                    );
                } else {
                    // Se non c'è domicilio, elimina eventuali record esistenti
                    $existingSoftDeleted->domicilio()->delete();
                }
                
                $message = 'Anagrafica ripristinata e aggiornata con successo';
            }

            Notification::make()
                ->warning()
                ->title('Attenzione')
                ->body("È stata trovata un'anagrafica eliminata con questa email. L'anagrafica è stata ripristinata.")
                ->persistent()
                ->actions([
                    \Filament\Notifications\Actions\Action::make('view')
                        ->label('Visualizza')
                        ->url(fn () => route('filament.admin.pages.gestione-contatti'))
                        ->button()
                        ->color('success'),
                    \Filament\Notifications\Actions\Action::make('close')
                        ->label('Chiudi')
                        ->close()
                        ->color('gray')
                        ->button(),
                ])
                ->send();

            $this->form->fill();
            return;
        }

        // Gestione anagrafica parziale esistente
        if ($existingPartial && isset($data['invite_user']) && $data['invite_user'] === true) {
            $token = Str::random(64);
            $existingPartial->update([
                'nome' => $data['nome'],
                'cognome' => $data['cognome'],
                'email' => $data['email'],
                'telefono' => $data['telefono'],
                'tipologia_utente' => $data['tipologia_utente'],
                'invitation_token' => $token,
                'invited_at' => now(),
                'is_complete' => false,
            ]);

            try {
                $url = route('complete.anagrafica', $token);
                Mail::to($data['email'])->send(new InviteToCompleteAnagrafica($existingPartial, $url));
                $message = 'Invito inviato con successo';
            } catch (\Exception $e) {
                $message = 'Errore durante l\'invio dell\'invito: ' . $e->getMessage();
            }

            Notification::make()
                ->success()
                ->title($message)
                ->send();

            $this->form->fill();
            return;
        }

        if (isset($data['invite_user']) && $data['invite_user'] === true) {
            $this->validate([
                'data.nome' => 'required|string|max:255',
                'data.cognome' => 'required|string|max:255',
                'data.email' => 'required|email|unique:anagrafiche,email',
                'data.telefono' => 'required|string|max:20',
                'data.tipologia_utente' => 'required|string',
            ]);
        
            try {
                $message = $this->sendInvitation($data);
            } catch (\Exception $e) {
                $message = 'Errore durante l\'invio dell\'invito: ' . $e->getMessage();
            }
        } else {
            $data['codice_fiscale'] = strtoupper($data['codice_fiscale']);
            $this->validate([
                'data.nome' => 'required|string|max:255',
                'data.cognome' => 'required|string|max:255',
                'data.email' => 'required|email|unique:anagrafiche,email,' . ($this->isEditing ? $this->anagrafica->id : 'NULL'),
                'data.telefono' => 'required|string|max:20',
                'data.genere' => 'required|in:M,F',
                'data.codice_fiscale' => [
                    'required',
                    'string',
                    'size:16',
                    'unique:anagrafiche,codice_fiscale,' . ($this->isEditing ? $this->anagrafica->id : 'NULL'),
                    function ($attribute, $value, $fail) use ($data) {
                        $cfGenerato = CodiceFiscale::generate(
                            $data['nome'],
                            $data['cognome'],
                            $data['data_nascita'],
                            $data['citta_nascita'],
                            $data['genere']
                        );
                        if ($value !== $cfGenerato) {
                            $fail('Il codice fiscale non corrisponde ai dati inseriti.');
                        }
                    },
                ],
                'data.data_nascita' => 'required|date',
                'data.citta_nascita' => 'required|string|max:255',
                'data.provincia_nascita' => 'required|string|max:2',
                'data.regione_nascita' => 'required|string|max:255',
                'data.indirizzo_residenza' => 'required|string|max:255',
                'data.numero_civico_residenza' => 'required|string|max:10',
                'data.cap_residenza' => 'required|string|size:5',
                'data.citta_residenza' => 'required|string|max:255',
                'data.provincia_residenza' => 'required|string|max:2',
                'data.tipologia_utente' => 'required|string',
            ]);

            // Rimuovi i campi del domicilio dai dati dell'anagrafica
            $anagraficaData = collect($data)->except([
                'has_domicilio',
                'indirizzo_domicilio',
                'numero_civico_domicilio',
                'cap_domicilio',
                'citta_domicilio',
                'provincia_domicilio',
            ])->toArray();

            if ($this->isEditing) {
                $this->anagrafica->update($anagraficaData);
            } else {
                $this->anagrafica = AnagraficaModel::create($anagraficaData);
            }

            // Gestione del domicilio
            if (!empty($data['has_domicilio'])) {
                $domicilioData = [
                    'indirizzo' => $data['indirizzo_domicilio'],
                    'numero_civico' => $data['numero_civico_domicilio'],
                    'cap' => $data['cap_domicilio'],
                    'citta' => $data['citta_domicilio'],
                    'provincia' => $data['provincia_domicilio'],
                ];
                
                $this->anagrafica->domicilio()->updateOrCreate(
                    ['anagrafica_id' => $this->anagrafica->id],
                    $domicilioData
                );
            } else {
                // Se non c'è domicilio, elimina eventuali record esistenti
                $this->anagrafica->domicilio()->delete();
            }

            $isComplete = $this->anagrafica->isComplete();
            $this->anagrafica->is_complete = $isComplete;
            $this->anagrafica->save();

            $message = $this->isEditing ? 'Anagrafica aggiornata con successo' : 'Anagrafica creata con successo';
        }

        Notification::make()
            ->success()
            ->title($message)
            ->send();

        if (!$this->isEditing) {
            $this->form->fill();
        }
    }

    public function saveAction(): \Filament\Forms\Components\Actions\Action
    {
        return \Filament\Forms\Components\Actions\Action::make('save')
            ->label(fn () => $this->isEditing ? 'Modifica anagrafica' : 'Aggiungi anagrafica')
            ->requiresConfirmation()
            ->action(function (array $arguments) {
                $data = $this->form->getState();
                $this->performSave($data);
            });
    }

    public function getContrattiCount()
    {
        return $this->isEditing ? $this->anagrafica->contratti()->count() : 0;
    }

    public function getTotaleCompensi()
    {
        return $this->anagrafica->contratti()->sum('compenso');
    }

    protected function getResidenzaAddressComponents($placeId)
    {
        try {
            $client = new Client();
            $response = $client->get("https://maps.googleapis.com/maps/api/place/details/json", [
                'query' => [
                    'place_id' => $placeId,
                    'key' => config('services.google.places_key'),
                    'language' => 'it'
                ]
            ]);

            $place = json_decode($response->getBody(), true);

            if (!isset($place['result']['address_components'])) {
                return null;
            }

            $components = [];
            foreach ($place['result']['address_components'] as $component) {
                if (in_array('locality', $component['types'])) {
                    $components['city'] = $component['long_name'];
                }
                if (in_array('postal_code', $component['types'])) {
                    $components['postal_code'] = $component['long_name'];
                }
                if (in_array('administrative_area_level_2', $component['types'])) {
                    $components['province'] = $component['short_name'];
                }
            }

            return $components;
        } catch (\Exception $e) {
            return null;
        }
    }

    protected function getDomicilioAddressComponents($placeId)
    {
        try {
            $client = new Client();
            $response = $client->get("https://maps.googleapis.com/maps/api/place/details/json", [
                'query' => [
                    'place_id' => $placeId,
                    'key' => config('services.google.places_key'),
                    'language' => 'it'
                ]
            ]);

            $place = json_decode($response->getBody(), true);

            if (!isset($place['result']['address_components'])) {
                return null;
            }

            $components = [];
            foreach ($place['result']['address_components'] as $component) {
                if (in_array('locality', $component['types'])) {
                    $components['city'] = $component['long_name'];
                }
                if (in_array('postal_code', $component['types'])) {
                    $components['postal_code'] = $component['long_name'];
                }
                if (in_array('administrative_area_level_2', $component['types'])) {
                    $components['province'] = $component['short_name'];
                }
            }

            return $components;
        } catch (\Exception $e) {
            return null;
        }
    }

    public function form(Form $form): Form
    {
        return $form
        ->schema([
            Toggle::make('invite_user')
            ->label("Invita l'utente a completare l'anagrafica")
            ->onIcon('heroicon-m-bolt')
            ->offIcon('heroicon-m-user')
            ->onColor('success')
            ->offColor('gray')
            ->default(false)
            ->dehydrated(true)
            ->live(),
            
            Tabs::make('Tabs')
                ->tabs([
                    Tabs\Tab::make('Scheda anagrafica')
                        ->schema([
                            Section::make('Dati principali')
                                ->description("Questi dati sono necessari per creare un'anagrafica sia completa che parziale")
                                ->icon('fas-address-card')
                                ->schema([
                                    TextInput::make('nome')->required(),
                                    TextInput::make('cognome')->required(),
                                    TextInput::make('telefono')
                                        ->label('Telefono')
                                        ->tel()
                                        ->required(),
                                    TextInput::make('email')
                                        ->label('Email')
                                        ->email()
                                        ->required(),
                                    Select::make('tipologia_utente')
                                        ->label('Tipologia Utente')
                                        ->relationship('tipologiaUtente', 'nome', fn ($query) => $query->orderBy('nome'))
                                        ->getOptionLabelFromRecordUsing(fn ($record) => $record->nome)
                                        ->required(),
                                ])
                                ->columns(2),
            
                                Section::make('Dati personali')
                                    ->icon('heroicon-s-user')
                                    ->schema([

                                        Select::make('genere')
                                            ->label('Genere')
                                            ->options([
                                                'M' => 'Maschio',
                                                'F' => 'Femmina',
                                            ])
                                            ->required()
                                            ->placeholder('Seleziona un\'opzione')
                                            ->afterStateHydrated(function (Select $component, $state) {
                                                if ($this->isEditing && $state) {
                                                    $component->state($state);
                                                }
                                            }),
          
                                        DatePicker::make('data_nascita')
                                            ->label('Data di nascita')
                                            ->required(),

                                        TextInput::make('codice_fiscale')
                                            ->label('Codice Fiscale')
                                            ->required()
                                            ->maxLength(16)
                                            ->live()
                                            ->afterStateUpdated(function ($state, callable $set, callable $get) {
                                                if ($get('invite_user')) {
                                                    return; // Non eseguire la validazione in modalità invito
                                                }
                                                
                                                $state = strtoupper($state); 
                                                $set('codice_fiscale', $state); 
                                                
                                                if (is_string($state) && strlen($state) == 16) {
                                                    $cf = new CodiceFiscale();
                                                    if (!$cf->tryParse($state)) {
                                                        $set('codice_fiscale_error', 'Il codice fiscale non è valido. ' . $cf->getError()->getMessage());
                                                    } else {
                                                        $set('codice_fiscale_error', null);
                                                        
                                                        $nome = $get('nome');
                                                        $cognome = $get('cognome');
                                                        $data_nascita = $get('data_nascita');
                                                        $citta_nascita = $get('citta_nascita');
                                                        $genere = $get('genere');
                                        
                                                        if ($nome && $cognome && $data_nascita && $citta_nascita && $genere) {
                                                            $cfGenerato = CodiceFiscale::generate($nome, $cognome, $data_nascita, $citta_nascita, $genere);
                                                            
                                                            if ($state !== $cfGenerato) {
                                                                $set('codice_fiscale_error', 'Il codice fiscale non corrisponde ai dati inseriti.');
                                                            }
                                                        }
                                                    }
                                                } else {
                                                    $set('codice_fiscale_error', null);
                                                }
                                            })
                                            ->helperText(fn ($get) => $get('codice_fiscale_error'))
                                            ->visible(fn (Get $get) => !$get('invite_user')),
                                
                                        Select::make('citta_nascita')
                                            ->label('Città di nascita')
                                            ->options(function () {
                                                return Comune::orderBy('nome')
                                                    ->get()
                                                    ->mapWithKeys(function ($comune) {
                                                        return [$comune->nome => $comune->nome_con_regione];
                                                    });
                                            })
                                            ->searchable()
                                            ->required()
                                            ->live()
                                            ->afterStateUpdated(function (callable $set, $state) {
                                                $comune = Comune::where('nome', $state)->first();
                                                if ($comune) {
                                                    $provincia = Provincia::where('nome', $comune->provincia)->first();
                                                    $set('provincia_nascita', $provincia ? $provincia->abbreviazione : null);
                                                    $set('regione_nascita', $comune->regione);
                                                }
                                            }),

                                        TextInput::make('provincia_nascita')
                                            ->label('Provincia di nascita')
                                            ->disabled()
                                            ->dehydrated(true)
                                            ->required(),

                                        TextInput::make('regione_nascita')
                                            ->label('Regione di nascita')
                                            ->disabled()
                                            ->dehydrated(true)
                                            ->required(),
                                    ])
                                    ->columns(3)
                                    ->visible(fn (Get $get) => !$get('invite_user')),
                                
                                    Section::make('Indirizzo di residenza')
                                        ->icon('fas-address-card')
                                        ->schema([
                                            GooglePlacesAutocomplete::make('indirizzo_residenza')
                                                ->label('Indirizzo')
                                                ->columnSpan(2)
                                                ->hint("Digita l'indirizzo e clicca la voce corretta per completare l'inserimento")
                                                ->required()
                                                ->afterStateUpdated(function ($state, callable $set) {
                                                    $addressComponents = $this->getResidenzaAddressComponents($state);
                                                    if ($addressComponents) {
                                                        $set('citta_residenza', $addressComponents['city'] ?? '');
                                                        $set('cap_residenza', $addressComponents['postal_code'] ?? '');
                                                        $set('provincia_residenza', $addressComponents['province'] ?? '');
                                                    }
                                                }),
                                            TextInput::make('numero_civico_residenza')
                                                ->label('Numero civico')
                                                ->required(),
                                            TextInput::make('cap_residenza')
                                                ->label('CAP')
                                                ->required()
                                                ->length(5),
                                            TextInput::make('citta_residenza')
                                                ->label('Città')
                                                ->required(),
                                            Select::make('provincia_residenza')
                                                ->label('Provincia')
                                                ->options(Provincia::pluck('nome', 'id'))
                                                ->searchable()
                                                ->required(),
                                        ])
                                        ->columns(2)
                                        ->visible(fn (Get $get) => !$get('invite_user')),

                                    Toggle::make('has_domicilio')
                                        ->label('Di avere domicilio diverso dalla residenza')
                                        ->live()
                                        ->visible(fn (Get $get) => !$get('invite_user')),

                                    Section::make('Domicilio')
                                        ->icon('fas-house')
                                        ->schema([
                                            GooglePlacesAutocomplete::make('indirizzo_domicilio')
                                                ->label('Indirizzo')
                                                ->columnSpan(2)
                                                ->hint("Digita l'indirizzo e clicca la voce corretta per completare l'inserimento")
                                                ->required()
                                                ->afterStateUpdated(function ($state, callable $set) {
                                                    $addressComponents = $this->getDomicilioAddressComponents($state);
                                                    if ($addressComponents) {
                                                        $set('citta_domicilio', $addressComponents['city'] ?? '');
                                                        $set('cap_domicilio', $addressComponents['postal_code'] ?? '');
                                                        $set('provincia_domicilio', $addressComponents['province'] ?? '');
                                                    }
                                                }),
                                            TextInput::make('numero_civico_domicilio')
                                                ->label('Numero civico')
                                                ->required(),
                                            TextInput::make('cap_domicilio')
                                                ->label('CAP')
                                                ->required()
                                                ->length(5),
                                            TextInput::make('citta_domicilio')
                                                ->label('Città')
                                                ->required(),
                                            Select::make('provincia_domicilio')
                                                ->label('Provincia')
                                                ->options(Provincia::pluck('nome', 'id'))
                                                ->searchable()
                                                ->required(),
                                        ])
                                        ->columns(2)
                                        ->visible(fn (Get $get) => $get('has_domicilio') && !$get('invite_user')),

                            Section::make('Rapporto di lavoro')
                                ->icon('heroicon-s-briefcase')
                                ->schema([
                                   

                                    Toggle::make('is_dipendente_pa_rapporto')
                                        ->label('Di essere legato/a da un rapporto di lavoro subordinato con la Pubblica Amministrazione')
                                        ->live(),

                                    Toggle::make('no_esclusioni')
                                        ->label('Di non rientrare nelle esclusioni soggettive previste dall\'art. 53 del D. Lgs. 165/2001 e ss.mm.ii')
                                        ->visible(fn (Get $get): bool => $get('is_dipendente_pa_rapporto'))
                                        ->live()
                                        ->onColor('success')
                                        ->offColor('gray'),

                                    TextInput::make('pa_denominazione_rapporto')
                                        ->label('Rilasciata da')
                                        ->visible(fn (Get $get): bool => $get('is_dipendente_pa_rapporto') && $get('no_esclusioni'))
                                        ->required(fn (Get $get): bool => $get('is_dipendente_pa_rapporto') && $get('no_esclusioni'))
                                        ->maxLength(255),

                                    TextInput::make('pa_autorizzazione_numero_rapporto')
                                        ->label('Autorizzazione n.')
                                        ->visible(fn (Get $get): bool => $get('is_dipendente_pa_rapporto') && $get('no_esclusioni'))
                                        ->required(fn (Get $get): bool => $get('is_dipendente_pa_rapporto') && $get('no_esclusioni'))
                                        ->maxLength(255),

                                    DatePicker::make('pa_autorizzazione_data_rapporto')
                                        ->label('del')
                                        ->visible(fn (Get $get): bool => $get('is_dipendente_pa_rapporto') && $get('no_esclusioni'))
                                        ->required(fn (Get $get): bool => $get('is_dipendente_pa_rapporto') && $get('no_esclusioni')),

                                    Toggle::make('pa_esclusione_tipo')
                                        ->label('Di rientrare nelle esclusioni soggettive previste dall\'art. 53 del D. Lgs. 165/2001 e ss.mm.ii')
                                        ->visible(fn (Get $get): bool => $get('is_dipendente_pa_rapporto'))
                                        ->live()
                                        ->onColor('danger')
                                        ->offColor('gray'),

                                    Select::make('pa_esclusione_dettagli')
                                        ->label('Tipo di esclusione')
                                        ->options([
                                            'tempo_definito' => 'Regime di impegno a tempo definito o tempo parziale con prestazione lavorativa non superiore al 50% di quella a tempo pieno',
                                            'formazione' => 'Art 53, comma 6 lett. f-bis, del Digs 165/01 e ss.mm.ii "attività di formazione diretta ai dipendenti della pubblica amministrazione"',
                                            'altro' => 'Altro'
                                        ])
                                        ->visible(fn (Get $get): bool => $get('is_dipendente_pa_rapporto') && $get('pa_esclusione_tipo'))
                                        ->live()
                                        ->required(fn (Get $get): bool => $get('is_dipendente_pa_rapporto') && $get('pa_esclusione_tipo')),

                                    TextInput::make('pa_esclusione_altro')
                                        ->label('Specificare altra esclusione')
                                        ->visible(fn (Get $get): bool => $get('is_dipendente_pa_rapporto') && $get('pa_esclusione_tipo') && $get('pa_esclusione_dettagli') === 'altro')
                                        ->required(fn (Get $get): bool => $get('is_dipendente_pa_rapporto') && $get('pa_esclusione_tipo') && $get('pa_esclusione_dettagli') === 'altro')
                                        ->maxLength(500),

                                    Toggle::make('is_pensionato_diretto')
                                        ->label('Di essere titolare di pensione diretta')
                                        ->live(),

                                    Toggle::make('is_senza_previdenza')
                                        ->label('Non iscritto ad alcuna forma previdenziale')
                                        ->onIcon('heroicon-m-user')
                                        ->offIcon('heroicon-m-user'),
                                ])
                                ->visible(fn (Get $get) => !$get('invite_user')),

                            Section::make('Dati fiscali e previdenziali')
                                ->icon('heroicon-s-credit-card')
                                ->schema([
                                    Select::make('tipologia_lavoro')
                                        ->label('Tipologia di lavoro')
                                        ->options([
                                            'lavoro_autonomo_professionale' => 'Lavoro autonomo professionale abituale',
                                            'regime_fiscale_vantaggio' => 'Contribuente nel nuovo regime fiscale di vantaggio',
                                            'lavoro_autonomo_occasionale' => 'Lavoro autonomo occasionale'
                                        ])
                                        ->required()
                                        ->live(),

                                    TextInput::make('partita_iva')
                                        ->label('Partita IVA')
                                        ->visible(fn (Get $get) => $get('tipologia_lavoro') === 'lavoro_autonomo_professionale')
                                        ->rules(['required_if:tipologia_lavoro,lavoro_autonomo_professionale'])
                                        ->rules(function ($get, $state) {
                                            if (preg_match('/^[A-Z]{2}/i', $state)) {
                                                return [new VatNumber(['format_only' => true])];
                                            }
                                            return ['regex:/^[0-9]{11}$/'];
                                        })
                                        ->live()
                                        ->hint('Se si inserisce il prefisso del paese (es. IT12345678901) verrà effettuata la verifica nel database VIES')
                                        ->afterStateUpdated(function ($state, callable $set) {
                                            if (!empty($state)) {
                                                if (preg_match('/^[A-Z]{2}/i', $state)) {
                                                    try {
                                                        $validator = new \Danielebarbaro\LaravelVatEuValidator\VatValidator();
                                                        $isValid = $validator->validate($state);
                                                        if ($isValid) {
                                                            $set('partita_iva_message', 'Partita IVA verificata nel database VIES ✓');
                                                            $set('partita_iva_message_color', 'success');
                                                        } else {
                                                            $set('partita_iva_message', 'Partita IVA non presente nel database VIES');
                                                            $set('partita_iva_message_color', 'warning');
                                                        }
                                                    } catch (\Exception $e) {
                                                        $set('partita_iva_message', null);
                                                    }
                                                } else {
                                                    $set('partita_iva_message', null);
                                                }
                                            } else {
                                                $set('partita_iva_message', null);
                                            }
                                        })
                                        ->helperText(fn (Get $get) => $get('partita_iva_message'))
                                        ->hintColor(fn (Get $get) => $get('partita_iva_message_color')),

                                    Select::make('aliquota_inps')
                                        ->label('Aliquota INPS')
                                        ->options([
                                            '33.72' => '33,72% (non iscritti ad altre forme pensionistiche)',
                                            '24.00' => '24,00% (pensionati o iscritti ad altra forma pensionistica)',
                                            '24.00_altra' => '24,00% (già iscritto ad altra cassa previdenziale)'
                                        ])
                                        ->visible(fn (Get $get) => $get('tipologia_lavoro') === 'lavoro_autonomo_occasionale'),

                                    TextInput::make('cassa_previdenziale')
                                        ->label('Cassa o ente previdenziale')
                                        ->visible(fn (Get $get) => $get('tipologia_lavoro') === 'lavoro_autonomo_professionale'),

                                    TextInput::make('albo_professionale')
                                        ->label('Albo o elenco professionale')
                                        ->visible(fn (Get $get) => $get('tipologia_lavoro') === 'lavoro_autonomo_professionale'),
                                ])
                                ->visible(fn (Get $get) => !$get('invite_user')),

                            Section::make('Dati pagamento')
                                ->icon('heroicon-s-banknotes')
                                ->schema([
                                    Select::make('tipo_pagamento')
                                        ->label('Tipo di pagamento')
                                        ->options([
                                            'bancario' => 'Accredito su c/c bancario',
                                            'postale' => 'Accredito su c/c postale'
                                        ])
                                        ->required()
                                        ->live(),

                                    TextInput::make('intestazione_conto')
                                        ->label('c/c intestato a')
                                        ->required()
                                        ->visible(fn (Get $get) => $get('tipo_pagamento') === 'bancario'),

                                    TextInput::make('iban')
                                        ->label('IBAN')
                                        ->required()
                                        ->visible(fn (Get $get) => $get('tipo_pagamento') === 'bancario')
                                        ->hint('Il conto corrente deve essere intestato o cointestato alla persona che ha ricevuto l\'incarico')
                                        ->rules(['iban']),

                                    TextInput::make('istituto_postale')
                                        ->label('Istituto postale')
                                        ->required()
                                        ->visible(fn (Get $get) => $get('tipo_pagamento') === 'postale'),
                                ])
                                ->visible(fn (Get $get) => !$get('invite_user')),

                            \Filament\Forms\Components\Actions::make([
                                $this->saveAction()
                            ])
                            ->columnSpan('full'),
                        ]),
                    Tabs\Tab::make('Contratti associati')
                    ->schema([
                        \Filament\Forms\Components\View::make('livewire.contratti-manager-wrapper')
                    ])
                    ->badge(fn () => $this->getContrattiCount())
                    ->visible(fn () => $this->isEditing)
                ,
                    Tabs\Tab::make('Riepilogo compensi')
                        ->schema([
                            \Filament\Forms\Components\View::make('livewire.riepilogo-compensi')
                        ])
                        ->visible(fn () => $this->isEditing)
                    ,
                    Tabs\Tab::make('CV Caricati')
                        ->visible(fn () => $this->isEditing)
                        ->badge(fn () => $this->anagrafica->cv_path ? 1 : 0)
                        ->schema([
                            Section::make('Curriculum Vitae')
                                ->schema([
                                    \Filament\Forms\Components\Placeholder::make('cv_current')
                                        ->content(function () {
                                            if (!$this->anagrafica->cv_path) {
                                                return new \Illuminate\Support\HtmlString('<span class="text-gray-500">Nessun CV caricato</span>');
                                            }
                                            
                                            if (!Storage::disk('local')->exists($this->anagrafica->cv_path)) {
                                                return new \Illuminate\Support\HtmlString('<span class="text-red-500">File non trovato</span>');
                                            }
                                            
                                            $info = [];
                                            if ($this->anagrafica->cv_original_filename) {
                                                $info[] = "<strong>Nome file:</strong> " . $this->anagrafica->cv_original_filename;
                                            }
                                            if ($this->anagrafica->cv_updated_at) {
                                                $info[] = "<strong>Caricato il:</strong> " . $this->anagrafica->cv_updated_at->format('d/m/Y H:i');
                                            }
                                            
                                            $downloadLink = view('filament.forms.components.cv-link-simple', [
                                                'path' => $this->anagrafica->cv_path,
                                                'label' => 'Visualizza CV',
                                                'anagrafica_id' => $this->anagrafica->id
                                            ])->render();
                                            
                                            return new \Illuminate\Support\HtmlString(
                                                $downloadLink . '<br><br>' . implode('<br>', $info)
                                            );
                                        })
                                        ->live()
                                        ->afterStateUpdated(function () {
                                            $this->anagrafica->refresh();
                                        }),
                                        
                                    \Filament\Forms\Components\FileUpload::make('new_cv')
                                        ->label('Carica nuovo CV')
                                        ->disk('local')
                                        ->directory('cv')
                                        ->acceptedFileTypes(['application/pdf'])
                                        ->maxSize(5120)
                                        ->downloadable()
                                        ->openable()
                                        ->preserveFilenames()
                                        ->afterStateUpdated(function ($state) {
                                            if (!$state || !is_object($state)) return;
                                            
                                            try {
                                                // Rimuovi il vecchio file se esiste
                                                if ($this->anagrafica->cv_path && Storage::disk('local')->exists($this->anagrafica->cv_path)) {
                                                    Storage::disk('local')->delete($this->anagrafica->cv_path);
                                                }

                                                $file = $state;
                                                
                                                // Crea un nome file più descrittivo
                                                $timestamp = now()->format('Y-m-d');
                                                $safeName = Str::slug(
                                                    "{$this->anagrafica->cognome}-{$this->anagrafica->nome}-cv-{$timestamp}"
                                                ) . '.' . $file->getClientOriginalExtension();
                                                
                                                $path = $file->storeAs('cv', $safeName, 'local');
                                                
                                                $this->anagrafica->cv_path = $path;
                                                $this->anagrafica->cv_original_filename = $file->getClientOriginalName();
                                                $this->anagrafica->cv_updated_at = now();
                                                $this->anagrafica->save();
                                                
                                                // Reset del campo file e forza il refresh
                                                $this->form->fill(['new_cv' => null]);
                                                $this->dispatch('cv-updated');
                                                
                                                Notification::make()
                                                    ->success()
                                                    ->title('CV aggiornato con successo')
                                                    ->duration(5000)
                                                    ->send();
                                            } catch (\Exception $e) {
                                                Notification::make()
                                                    ->danger()
                                                    ->title('Errore durante il salvataggio del CV')
                                                    ->body($e->getMessage())
                                                    ->duration(5000)
                                                    ->send();
                                            }
                                        })
                                ])
                                ->columns(1)
                        ])
                ])
                ->activeTab(1)
                ->persistTabInQueryString()
        ])
        ->statePath('data')
        ->model($this->anagrafica);
    }

    public function render()
    {
        return view('livewire.create-anagrafica');
    }
}
