<?php

namespace App\Livewire;

use Livewire\Component;
use App\Models\Contratto;
use App\Models\TipoContratto;
use Filament\Forms;
use Filament\Forms\Concerns\InteractsWithForms;
use Filament\Forms\Contracts\HasForms;
use Filament\Tables;
use Filament\Tables\Concerns\InteractsWithTable;
use Filament\Tables\Contracts\HasTable;

use Filament\Tables\Actions\ViewAction;
use Filament\Tables\Actions\EditAction;
use Filament\Tables\Actions\DeleteAction;
use Illuminate\Support\HtmlString;

class ContrattiManager extends Component implements HasForms, HasTable
{
    use InteractsWithForms;
    use InteractsWithTable;

    public $anagrafica;

    public function mount($anagrafica)
    {
        $this->anagrafica = $anagrafica;
    }
    
    protected function getContrattoForm(): array
    {
        return [
            Forms\Components\Select::make('tipo_contratto_id')
                ->label('Tipo Contratto')
                ->options(TipoContratto::pluck('nome', 'id'))
                ->required(),
            Forms\Components\DatePicker::make('data_inizio')
                ->label('Data Inizio')
                ->required(),
            Forms\Components\DatePicker::make('data_fine')
                ->label('Data Fine'),
            Forms\Components\Toggle::make('lavoro_subordinato_pa')
                ->label('Lavoro Subordinato PA')
                ->reactive(),
            Forms\Components\TextInput::make('denominazione_pa')
                ->label('Denominazione PA')
                ->visible(fn (callable $get) => $get('lavoro_subordinato_pa')),
            Forms\Components\TextInput::make('indirizzo_pa')
                ->label('Indirizzo PA')
                ->visible(fn (callable $get) => $get('lavoro_subordinato_pa')),
        ];
    }

    protected function getFormSchema(): array
    {
        return [
            Forms\Components\Select::make('tipo_contratto_id')
                ->label('Tipo Contratto')
                ->options(TipoContratto::pluck('nome', 'id'))
                ->required(),
            Forms\Components\DatePicker::make('data_inizio')
                ->required(),
            Forms\Components\DatePicker::make('data_fine'),
        ];
    }

    protected function getTableQuery()
    {
        $query = Contratto::query()->where('anagrafica_id', $this->anagrafica->id);
        $results = $query->get();
        return $query;
    }

    public function hasContratti()
    {
        return $this->getTableQuery()->count() > 0;
    }

    protected function getTableColumns(): array
    {
        return [
            Tables\Columns\TextColumn::make('id')
                ->label('ID'),
            Tables\Columns\TextColumn::make('tipoContratto.nome')
                ->label('Tipo Contratto'),
            Tables\Columns\TextColumn::make('data_inizio')
                ->date(),
            Tables\Columns\TextColumn::make('data_fine')
                ->date(),
        ];
    }

    protected function getTableActions(): array
    {
        return [
            ViewAction::make()
            ->modalHeading('Dettagli Contratto')
            ->modalContent(function (Contratto $record): HtmlString {
                return new HtmlString(view(
                    'contratti.details',
                    ['contratto' => $record]
                )->render());
            })
            ->modalWidth('lg'),
            EditAction::make()
                ->modalHeading('Modifica Contratto')
                ->form($this->getContrattoForm())
                ->mutateFormDataUsing(function (array $data): array {
                    return $data;
                })
                ->action(function (Contratto $record, array $data): void {
                    $record->update($data);
            }),
            DeleteAction::make(),
        ];
    }

    protected function getTableHeaderActions(): array
    {
        return [
            Tables\Actions\CreateAction::make()
                ->mutateFormDataUsing(function (array $data): array {
                    $data['anagrafica_id'] = $this->anagrafica->id;
                    return $data;
                }),
        ];
    }

    public function render()
    {
        $data = [
            'records' => $this->getTableQuery()->get(),
        ];

        return view('livewire.contratti-manager', $data);
    }
}