<?php

namespace App\Livewire;

use App\Models\Candidature;
use Filament\Forms\Components\FileUpload;
use Filament\Forms\Components\Section;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Checkbox;
use Filament\Forms\Concerns\InteractsWithForms;
use Filament\Forms\Contracts\HasForms;
use Filament\Forms\Form;
use Filament\Notifications\Notification;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Log;
use Livewire\Component;
use App\Mail\CandidatureCompleted;

class CompleteCandidature extends Component implements HasForms
{
    use InteractsWithForms;

    public ?array $data = [];
    public Candidature $candidature;
    public bool $isCompleted = false;

    public function mount($token)
    {
        $this->candidature = Candidature::where(function($query) use ($token) {
            $query->where('invitation_token', $token)
                ->orWhere(function($q) use ($token) {
                    $q->where('is_complete', true)
                        ->whereNull('invitation_token');
                });
        })->firstOrFail();

        if ($this->candidature->is_complete || is_null($this->candidature->invitation_token)) {
            $this->isCompleted = true;
            return;
        }

        $this->form->fill([
            'nome' => $this->candidature->nome,
            'cognome' => $this->candidature->cognome,
            'email' => $this->candidature->email,
        ]);
    }

    public function form(Form $form): Form
    {
        return $form
            ->schema([
                Section::make('Dati Candidatura')
                    ->schema([
                        TextInput::make('nome')
                            ->required()
                            ->disabled(),
                        TextInput::make('cognome')
                            ->required()
                            ->disabled(),
                        TextInput::make('email')
                            ->email()
                            ->required()
                            ->disabled(),
                        FileUpload::make('cv')
                            ->label('Curriculum Vitae')
                            ->required()
                            ->acceptedFileTypes(['application/pdf'])
                            ->maxSize(5120)
                            ->directory('cv')
                            ->visibility('public')
                            ->storeFileNamesIn('original_filename'),
                    ])
                    ->columns(2),

                Section::make('Privacy')
                    ->schema([
                        Checkbox::make('privacy_accepted')
                            ->label('Accetto la Privacy Policy')
                            ->helperText('Acconsento al trattamento dei miei dati personali secondo la Privacy Policy')
                            ->required()
                            ->live(),
                    ])
                    ->columnSpanFull(),
            ])
            ->statePath('data');
    }

    public function save()
    {
        $data = $this->form->getState();
        
        if (!$data['privacy_accepted']) {
            Notification::make()
                ->title('Errore')
                ->body('Devi accettare la Privacy Policy per procedere.')
                ->danger()
                ->send();
            return;
        }

        // Salva il CV
        if (isset($data['cv']) && $data['cv']) {
            $this->candidature->cv_path = $data['cv'];
            
            // Log per debug
            \Illuminate\Support\Facades\Log::info('Saving CV path: ' . $data['cv']);
        }

        $this->candidature->invitation_token = null;
        $this->candidature->is_complete = true;
        $this->candidature->privacy_accepted = true;
        $this->candidature->privacy_accepted_at = now();
        $this->candidature->save();

        // Invia email di notifica
        $staffEmail = config('mail.staff_address');
        if ($staffEmail) {
            try {
                Mail::to($staffEmail)->send(new CandidatureCompleted($this->candidature));
            } catch (\Exception $e) {
                // Log l'errore ma non bloccare il processo
                \Illuminate\Support\Facades\Log::error('Errore nell\'invio dell\'email di notifica: ' . $e->getMessage());
            }
        }
        
        $this->isCompleted = true;

        Notification::make()
            ->title('Candidatura completata con successo')
            ->success()
            ->send();
    }

    public function render()
    {
        return view('livewire.complete-candidature', [
            'isCompleted' => $this->isCompleted,
        ]);
    }
} 