<?php

namespace App\Http\Controllers;

use App\Models\Contratto;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Symfony\Component\HttpFoundation\Response;

class ContrattoController extends Controller
{
    public function downloadPdf($id)
    {
        $contratto = Contratto::findOrFail($id);

        if (!$contratto->pdf_path || !Storage::disk('public')->exists($contratto->pdf_path)) {
            abort(404, 'PDF non trovato');
        }

        return Storage::disk('public')->download(
            $contratto->pdf_path,
            'contratto_' . $contratto->id . '.pdf',
            [
                'Content-Type' => 'application/pdf',
                'Content-Disposition' => 'attachment; filename="contratto_' . $contratto->id . '.pdf"'
            ]
        );
    }

    public function downloadContrattoFirmato($id)
    {
        $contratto = Contratto::findOrFail($id);

        if ($contratto->yousign_status !== 'completed') {
            abort(404, 'Contratto non ancora firmato');
        }

        // Qui dovresti implementare la logica per scaricare il PDF firmato da Yousign
        // Per ora restituiamo il PDF non firmato
        if (!$contratto->pdf_path || !Storage::disk('public')->exists($contratto->pdf_path)) {
            abort(404, 'PDF non trovato');
        }

        return Storage::disk('public')->download(
            $contratto->pdf_path,
            'contratto_firmato_' . $contratto->id . '.pdf',
            [
                'Content-Type' => 'application/pdf',
                'Content-Disposition' => 'attachment; filename="contratto_firmato_' . $contratto->id . '.pdf"'
            ]
        );
    }

    protected function generatePdf(Contratto $contratto)
    {
        $tipoContratto = $contratto->tipoContratto;
        $anagrafica = $contratto->anagrafica;

        // Mappa dei profili di contratto
        $templateMap = [
            1 => 'pdf.contratto-profilo-1',  // Personale strutturato
            2 => 'pdf.contratto-profilo-2',  // Docenza
            3 => 'pdf.contratto-profilo-3',  // Tutoraggio
            4 => 'pdf.contratto-profilo-4',  // Contratto di lavoro dipendente
            5 => 'pdf.contratto-profilo-5',  // Collaborazione occasionale
            6 => 'pdf.contratto-profilo-6',  // COCOCO
            7 => 'pdf.contratto-profilo-7',  // Collaborazione autonoma (P.IVA)
        ];

        // Se esiste un template specifico per questo tipo di contratto, usalo
        if (isset($templateMap[$tipoContratto->id])) {
            return \Barryvdh\DomPDF\Facade\Pdf::loadView($templateMap[$tipoContratto->id], [
                'contratto' => $contratto,
                'anagrafica' => $anagrafica,
                'tipoContratto' => $tipoContratto,
            ])
                ->setPaper('a4')
                ->output();
        }

        // Se il tipo contratto ha slug 'test', usa il template di test
        if ($tipoContratto->slug === 'test') {
            // Ottieni i dati del secondo firmatario
            $secondiFirmatari = [
                'direttore' => [
                    'first_name' => 'Mario',
                    'last_name' => 'Rossi',
                    'title' => 'Il Direttore Generale'
                ],
                'presidente' => [
                    'first_name' => 'Massimiliano',
                    'last_name' => 'Bizzocchi',
                    'title' => 'Il Presidente'
                ],
                'rettore' => [
                    'first_name' => 'Carlo',
                    'last_name' => 'Tasso',
                    'title' => 'Il Rettore'
                ]
            ];

            $secondoFirmatarioInfo = $secondiFirmatari[$contratto->secondo_firmatario] ?? $secondiFirmatari['direttore'];

            return \Barryvdh\DomPDF\Facade\Pdf::loadView('pdf.contratto-test', [
                'secondoFirmatarioInfo' => $secondoFirmatarioInfo
            ])
                ->setPaper('a4')
                ->output();
        }

        // Per tutti gli altri tipi di contratto, usa il template generico
        $viewData = [
            'contratto' => $contratto,
            'anagrafica' => $anagrafica,
            'tipoContratto' => $tipoContratto,
        ];

        return \Barryvdh\DomPDF\Facade\Pdf::loadView('pdf.contratto', $viewData)
            ->setPaper('a4')
            ->output();
    }
}