<?php

namespace App\Http\Controllers;

use App\Models\Anagrafica;
use App\Models\Contratto;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use App\Jobs\SendStaffAnagraficaCompletedNotification;
use Illuminate\Validation\Rule;
use Filament\Notifications\Notification;

class AnagraficaController extends Controller
{
    public function showCompleteForm($token)
    {
        try {
            // Prima cerchiamo l'anagrafica con il token
            $anagrafica = Anagrafica::where('invitation_token', $token)->first();

            // Se non troviamo l'anagrafica con il token, cerchiamo tra quelle completate
            if (!$anagrafica) {
                $anagrafica = Anagrafica::where(function($query) use ($token) {
                    $query->whereNull('invitation_token')
                          ->where('is_complete', true);
                })->latest()->first();

                if ($anagrafica) {
                    return view('pages.anagrafica-completed', [
                        'message' => 'Questa anagrafica è già stata completata con successo.',
                        'type' => 'success'
                    ]);
                }

                return view('pages.anagrafica-completed', [
                    'message' => 'Il link non è più valido o l\'anagrafica non è stata trovata.',
                    'type' => 'error'
                ]);
            }
            
            return view('pages.completa-anagrafica', compact('token'));
        } catch (\Exception $e) {
            Log::error('Errore nel completamento anagrafica:', [
                'token' => $token,
                'error' => $e->getMessage()
            ]);

            return view('pages.anagrafica-completed', [
                'message' => 'Si è verificato un errore durante l\'accesso all\'anagrafica.',
                'type' => 'error'
            ]);
        }
    }

    public function saveCompleteForm(Request $request, $token)
    {
        Log::info('saveCompleteForm started for Anagrafica', ['token' => $token]);

        $anagrafica = Anagrafica::where('invitation_token', $token)->firstOrFail();
        Log::info('Anagrafica found', ['anagrafica_id' => $anagrafica->id]);

        $validatedData = $request->validate([
            'nome' => 'required',
            'cognome' => 'required',
            'email' => [
                'required',
                'email',
                Rule::unique('anagrafiche')->ignore($anagrafica->id)->whereNull('deleted_at')
            ],
            'telefono' => 'required',
            'codice_fiscale' => [
                'required',
                Rule::unique('anagrafiche')->ignore($anagrafica->id)->whereNull('deleted_at')
            ],
        ]);
        
        $anagrafica->update($validatedData);
        $anagrafica->invitation_token = null;
        $anagrafica->is_complete = true;
        $anagrafica->save();
        
        SendStaffAnagraficaCompletedNotification::dispatch($anagrafica);
        
        return redirect()->route('home')->with('success', 'Anagrafica completata con successo');
    }

    protected function validateAnagrafica($request, $id = null)
    {
        try {
            return $request->validate([
                'nome' => 'required',
                'cognome' => 'required',
                'email' => [
                    'required',
                    'email',
                    Rule::unique('anagrafiche')->ignore($id)->whereNull('deleted_at')
                ],
                'telefono' => 'required',
                'codice_fiscale' => [
                    'required',
                    Rule::unique('anagrafiche')->ignore($id)->whereNull('deleted_at')
                ],
            ]);
        } catch (\Illuminate\Validation\ValidationException $e) {
            // Se l'email esiste già, controlla se è un record soft-deleted
            if (isset($e->errors()['email'])) {
                $existingSoftDeleted = Anagrafica::withTrashed()
                    ->where('email', $request->email)
                    ->whereNotNull('deleted_at')
                    ->first();

                if ($existingSoftDeleted) {
                    return $request->all(); // Permetti l'operazione per record soft-deleted
                }
            }
            
            throw $e; // Rilancia l'eccezione se non è un caso di record soft-deleted
        }
    }

    public function store(Request $request)
    {
        try {
            // Prima controlla se esiste un'anagrafica soft-deleted
            $existingSoftDeleted = Anagrafica::withTrashed()
                ->where(function ($query) use ($request) {
                    $query->where('email', $request->email)
                          ->orWhere('codice_fiscale', $request->codice_fiscale);
                })
                ->whereNotNull('deleted_at')
                ->first();

            if ($existingSoftDeleted) {
                // Se esiste, ripristina e aggiorna
                $existingSoftDeleted->restore();
                $existingSoftDeleted->update($request->all());
                
                Notification::make()
                    ->warning()
                    ->title('Attenzione')
                    ->body("È stata trovata un'anagrafica eliminata con questa email/codice fiscale. L'anagrafica è stata ripristinata e aggiornata.")
                    ->persistent()
                    ->actions([
                        \Filament\Notifications\Actions\Action::make('view')
                            ->label('Visualizza')
                            ->url(fn () => "/admin/anagrafiche/{$existingSoftDeleted->id}/edit")
                            ->button(),
                    ])
                    ->send();

                return redirect()->back();
            }

            // Se non esiste un record soft-deleted, procedi con la validazione normale
            $data = $this->validateAnagrafica($request);
            
            // Crea una nuova anagrafica
            $anagrafica = Anagrafica::create($data);

            Notification::make()
                ->success()
                ->title('Successo')
                ->body('Anagrafica creata con successo')
                ->send();

            return redirect()->back();

        } catch (\Exception $e) {
            Notification::make()
                ->danger()
                ->title('Errore')
                ->body('Si è verificato un errore durante la creazione dell\'anagrafica: ' . $e->getMessage())
                ->persistent()
                ->send();

            return redirect()->back();
        }
    }
}