<?php

namespace App\Filament\Resources;

use App\Filament\Resources\CandidatureResource\Pages;
use App\Models\Candidature;
use Filament\Forms;
use Filament\Forms\Form;
use Filament\Resources\Resource;
use Filament\Tables;
use Filament\Tables\Table;
use Illuminate\Support\Facades\Mail;
use App\Mail\CandidatureInvitation;
use App\Mail\CandidatureCompleted;
use Filament\Notifications\Notification;
use Illuminate\Support\Facades\Storage;
use ZipArchive;
use Illuminate\Support\Facades\Response;
use Illuminate\Support\Facades\Auth;
use Filament\Facades\Filament;
use App\Models\PropostaCorso;
use App\Mail\CandidatureSelectionClosed;

class CandidatureResource extends Resource
{
    protected static ?string $model = Candidature::class;

    protected static ?string $navigationIcon = 'heroicon-o-user-group';

    protected static ?string $navigationGroup = 'Gestione Corsi';

    protected static ?int $navigationSort = 2;

    public static function getNavigationGroup(): ?string
    {
        return __('Gestione Corsi');
    }

    public static function getNavigationLabel(): string
    {
        return __('Candidature');
    }

    public static function getPluralModelLabel(): string
    {
        return __('Candidature');
    }

    public static function getModelLabel(): string
    {
        return __('Candidatura');
    }

    public static function form(Form $form): Form
    {
        return $form
            ->schema([
                Forms\Components\Section::make()
                    ->schema([
                        Forms\Components\TextInput::make('nome')
                            ->required()
                            ->maxLength(255),
                        Forms\Components\TextInput::make('cognome')
                            ->required()
                            ->maxLength(255),
                        Forms\Components\TextInput::make('email')
                            ->email()
                            ->required()
                            ->maxLength(255),
                        Forms\Components\Select::make('proposta_corso_id')
                            ->relationship('propostaCorso', 'titolo')
                            ->searchable()
                            ->preload()
                            ->required(),
                        Forms\Components\Toggle::make('is_approved')
                            ->label('Approvato')
                            ->helperText('Candidatura approvata dal CDA')
                            ->visible(fn () => Filament::auth()->user()->hasRole(['super_admin', 'segreteria_organi']))
                            ->afterStateUpdated(function ($record, $state) {
                                if ($record) {
                                    $record->approved_at = $state ? now() : null;
                                    $record->save();
                                }
                            }),
                    ])
                    ->columns(2),
            ]);
    }

    public static function table(Table $table): Table
    {
        return $table
            ->columns([
                Tables\Columns\TextColumn::make('nome')
                    ->searchable(),
                Tables\Columns\TextColumn::make('cognome')
                    ->searchable(),
                Tables\Columns\TextColumn::make('email')
                    ->searchable(),
                Tables\Columns\TextColumn::make('propostaCorso.titolo')
                    ->label('Corso')
                    ->searchable(),
                Tables\Columns\IconColumn::make('is_complete')
                    ->boolean()
                    ->label('Completato'),
                Tables\Columns\IconColumn::make('is_approved')
                    ->boolean()
                    ->label('Approvato')
                    ->trueIcon('heroicon-o-check-badge')
                    ->falseIcon('heroicon-o-x-mark')
                    ->visible(fn () => Filament::auth()->user()->hasRole(['super_admin', 'segreteria_organi'])),
                Tables\Columns\TextColumn::make('approved_at')
                    ->label('Data Approvazione')
                    ->dateTime()
                    ->sortable()
                    ->visible(fn () => Filament::auth()->user()->hasRole(['super_admin', 'segreteria_organi'])),
                Tables\Columns\TextColumn::make('created_at')
                    ->dateTime()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                Tables\Filters\SelectFilter::make('proposta_corso_id')
                    ->relationship('propostaCorso', 'titolo')
                    ->label('Corso')
                    ->searchable()
                    ->preload(),
                Tables\Filters\TernaryFilter::make('is_complete')
                    ->label('Completato'),
                Tables\Filters\TernaryFilter::make('is_approved')
                    ->label('Approvato')
                    ->visible(fn () => Filament::auth()->user()->hasRole(['super_admin', 'segreteria_organi'])),
            ])
            ->headerActions([
                Tables\Actions\Action::make('download_all_cvs')
                    ->label('Scarica tutti i CV')
                    ->icon('heroicon-o-archive-box-arrow-down')
                    ->color('primary')
                    ->requiresConfirmation()
                    ->modalHeading('Scarica CV')
                    ->modalDescription('Stai per scaricare i CV di tutti i candidati filtrati. Vuoi procedere?')
                    ->action(function ($livewire) {
                        // Get the filtered query from the table
                        $query = $livewire->getFilteredTableQuery();
                        $records = $query->get();

                        if ($records->isEmpty()) {
                            Notification::make()
                                ->title('Nessun candidato trovato')
                                ->warning()
                                ->send();
                            return;
                        }

                        $zip = new ZipArchive();
                        $zipFileName = 'cv_candidati_' . now()->format('Y-m-d_His') . '.zip';
                        $zipPath = storage_path('app/public/temp/' . $zipFileName);
                        
                        // Create temp directory if it doesn't exist
                        if (!Storage::exists('public/temp')) {
                            Storage::makeDirectory('public/temp');
                        }
                        
                        if ($zip->open($zipPath, ZipArchive::CREATE) === TRUE) {
                            // Create CSV file for summary
                            $csvContent = "Nome,Cognome,Email,Corso,CV Caricato\n";
                            $hasCvs = false;
                            
                            foreach ($records as $record) {
                                // Add to CSV regardless of CV presence
                                $csvContent .= sprintf(
                                    "%s,%s,%s,%s,%s\n",
                                    $record->nome,
                                    $record->cognome,
                                    $record->email,
                                    $record->propostaCorso->titolo,
                                    $record->cv_path ? 'Sì' : 'No'
                                );
                                
                                // Add CV to ZIP if exists
                                if ($record->cv_path) {
                                    $originalFileName = basename($record->cv_path);
                                    $newFileName = "{$record->cognome}_{$record->nome}_cv." . pathinfo($originalFileName, PATHINFO_EXTENSION);
                                    
                                    // Convert the storage URL to a real path
                                    $cvPath = str_replace('/storage/', '', $record->cv_path);
                                    if (Storage::disk('public')->exists($cvPath)) {
                                        $zip->addFile(storage_path('app/public/' . $cvPath), "cv/{$newFileName}");
                                        $hasCvs = true;
                                    }
                                }
                            }
                            
                            // Add CSV to ZIP
                            $zip->addFromString('riepilogo.csv', $csvContent);
                            $zip->close();
                            
                            if (!$hasCvs) {
                                Notification::make()
                                    ->title('Nessun CV trovato tra i candidati filtrati')
                                    ->warning()
                                    ->send();
                            }
                            
                            // Return download response
                            return Response::download($zipPath)->deleteFileAfterSend();
                        }
                        
                        Notification::make()
                            ->title('Errore durante la creazione del file ZIP')
                            ->danger()
                            ->send();
                            
                        return null;
                    }),
                Tables\Actions\Action::make('close_selection')
                    ->label('Chiudi Selezione')
                    ->icon('heroicon-o-lock-closed')
                    ->color('success')
                    ->visible(fn () => Filament::auth()->user()->hasRole(['super_admin', 'segreteria_organi']))
                    ->requiresConfirmation()
                    ->modalHeading('Chiudi Selezione Candidature')
                    ->modalDescription('Questa azione chiuderà la selezione per il corso selezionato e invierà una notifica al manager didattico con il riepilogo dei candidati approvati. Vuoi procedere?')
                    ->modalSubmitActionLabel('Sì, chiudi selezione')
                    ->form([
                        Forms\Components\Select::make('proposta_corso_id')
                            ->label('Corso')
                            ->relationship('propostaCorso', 'titolo')
                            ->required()
                            ->searchable()
                            ->preload(),
                        Forms\Components\Select::make('manager_email')
                            ->label('Email Manager Didattico')
                            ->options(function () {
                                return \App\Models\User::role('manager_didattico')
                                    ->pluck('email', 'email')
                                    ->toArray();
                            })
                            ->required()
                            ->searchable()
                            ->preload(),
                    ])
                    ->action(function (array $data, $livewire) {
                        $corso = PropostaCorso::find($data['proposta_corso_id']);
                        
                        if ($corso->selection_closed) {
                            Notification::make()
                                ->title('La selezione per questo corso è già stata chiusa')
                                ->warning()
                                ->send();
                            return;
                        }

                        // Get approved candidates for this course
                        $approvedCandidates = Candidature::where('proposta_corso_id', $corso->id)
                            ->where('is_approved', true)
                            ->get();

                        if ($approvedCandidates->isEmpty()) {
                            Notification::make()
                                ->title('Non ci sono candidati approvati per questo corso')
                                ->warning()
                                ->send();
                            return;
                        }

                        // Create partial anagrafiche for approved candidates
                        foreach ($approvedCandidates as $candidate) {
                            // Check if anagrafica already exists
                            $existingAnagrafica = \App\Models\Anagrafica::where('email', $candidate->email)
                                ->withTrashed()
                                ->first();

                            // Prepare anagrafica data
                            $anagraficaData = [
                                'nome' => $candidate->nome,
                                'cognome' => $candidate->cognome,
                                'email' => $candidate->email,
                                'is_complete' => false
                            ];

                            // Add CV data if exists
                            if ($candidate->cv_path) {
                                $anagraficaData['cv_path'] = $candidate->cv_path;
                                $anagraficaData['cv_original_filename'] = basename($candidate->cv_path);
                                $anagraficaData['cv_updated_at'] = now();
                            }

                            if ($existingAnagrafica) {
                                if ($existingAnagrafica->trashed()) {
                                    $existingAnagrafica->restore();
                                }
                                // Update existing anagrafica
                                $existingAnagrafica->update($anagraficaData);
                            } else {
                                // Create new partial anagrafica
                                \App\Models\Anagrafica::create($anagraficaData);
                            }

                            // Delete the candidature
                            $candidate->delete();
                        }

                        // Close the selection
                        $corso->update([
                            'selection_closed' => true,
                            'selection_closed_at' => now(),
                        ]);

                        // Send email to manager with only the names for the notification
                        $candidatesForEmail = $approvedCandidates->map(fn ($candidate) => [
                            'nome' => $candidate->nome,
                            'cognome' => $candidate->cognome,
                            'email' => $candidate->email,
                        ])->toArray();

                        Mail::to($data['manager_email'])->send(
                            new CandidatureSelectionClosed($corso, $candidatesForEmail)
                        );

                        Notification::make()
                            ->title('Selezione chiusa con successo')
                            ->success()
                            ->send();

                        // Refresh the table
                        $livewire->dispatch('filament-tables::reloadTable');
                    }),
            ])
            ->actions([
                Tables\Actions\Action::make('download_cv')
                    ->label('Scarica CV')
                    ->icon('heroicon-o-document-arrow-down')
                    ->color('success')
                    ->url(fn (Candidature $record) => $record->cv_path ? Storage::url($record->cv_path) : null)
                    ->openUrlInNewTab()
                    ->visible(fn (Candidature $record) => $record->cv_path !== null),
                Tables\Actions\Action::make('send_invitation')
                    ->label('Invita')
                    ->icon('heroicon-o-envelope')
                    ->requiresConfirmation()
                    ->visible(fn (Candidature $record) => !$record->is_complete)
                    ->action(function (Candidature $record) {
                        $record->generateInvitationToken();
                        Mail::to($record->email)->send(new CandidatureInvitation($record));
                        
                        Notification::make()
                            ->title('Invito inviato con successo')
                            ->success()
                            ->send();
                    }),
                Tables\Actions\ViewAction::make(),
                Tables\Actions\EditAction::make(),
                Tables\Actions\DeleteAction::make(),
            ])
            ->bulkActions([
                Tables\Actions\BulkActionGroup::make([
                    Tables\Actions\DeleteBulkAction::make(),
                    Tables\Actions\BulkAction::make('download_cvs')
                        ->label('Scarica CV')
                        ->icon('heroicon-o-archive-box-arrow-down')
                        ->requiresConfirmation()
                        ->action(function ($records) {
                            $zip = new ZipArchive();
                            $zipFileName = 'cv_candidati_' . now()->format('Y-m-d_His') . '.zip';
                            $zipPath = storage_path('app/public/temp/' . $zipFileName);
                            
                            // Create temp directory if it doesn't exist
                            if (!Storage::exists('public/temp')) {
                                Storage::makeDirectory('public/temp');
                            }
                            
                            if ($zip->open($zipPath, ZipArchive::CREATE) === TRUE) {
                                // Create CSV file for summary
                                $csvContent = "Nome,Cognome,Email,Corso,CV Caricato\n";
                                $hasCvs = false;
                                
                                foreach ($records as $record) {
                                    // Add to CSV regardless of CV presence
                                    $csvContent .= sprintf(
                                        "%s,%s,%s,%s,%s\n",
                                        $record->nome,
                                        $record->cognome,
                                        $record->email,
                                        $record->propostaCorso->titolo,
                                        $record->cv_path ? 'Sì' : 'No'
                                    );
                                    
                                    // Add CV to ZIP if exists
                                    if ($record->cv_path) {
                                        $originalFileName = basename($record->cv_path);
                                        $newFileName = "{$record->cognome}_{$record->nome}_cv." . pathinfo($originalFileName, PATHINFO_EXTENSION);
                                        
                                        // Convert the storage URL to a real path
                                        $cvPath = str_replace('/storage/', '', $record->cv_path);
                                        if (Storage::disk('public')->exists($cvPath)) {
                                            $zip->addFile(storage_path('app/public/' . $cvPath), "cv/{$newFileName}");
                                            $hasCvs = true;
                                        }
                                    }
                                }
                                
                                // Add CSV to ZIP
                                $zip->addFromString('riepilogo.csv', $csvContent);
                                $zip->close();
                                
                                if (!$hasCvs) {
                                    Notification::make()
                                        ->title('Nessun CV trovato tra i candidati selezionati')
                                        ->warning()
                                        ->send();
                                }
                                
                                // Return download response
                                return Response::download($zipPath)->deleteFileAfterSend();
                            }
                            
                            Notification::make()
                                ->title('Errore durante la creazione del file ZIP')
                                ->danger()
                                ->send();
                                
                            return null;
                        }),
                ]),
            ]);
    }

    public static function getRelations(): array
    {
        return [
            //
        ];
    }

    public static function getPages(): array
    {
        return [
            'index' => Pages\ListCandidatures::route('/'),
            'create' => Pages\CreateCandidature::route('/create'),
            'edit' => Pages\EditCandidature::route('/{record}/edit'),
        ];
    }
} 